/*
 * Embed a tag expression in a class with proper copy semantics
 * Plus some filters based on tag expressions
 *
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tagcoll/expression.h>

#include <tagcoll/tagexpr/TagexprParser.h>

using namespace std;

namespace tagcoll {

class TagexprTrue : public ExpressionImpl
{
public:
	TagexprTrue() {}
	virtual ~TagexprTrue() {}

	virtual std::string format() const { return "*"; }

	virtual bool eval(const TagexprContext& context) const { return true; }
	virtual bool eval(const std::set<std::string>& tags) const { return true; }
};

class TagexprTag : public ExpressionImpl
{
protected:
	std::string _tag;
	
public:
	TagexprTag(const std::string& tag) : _tag(tag) {}
	virtual ~TagexprTag() {}

	virtual std::string format() const { return _tag; }

	virtual bool eval(const TagexprContext& context) const
	{
		return context.eval(_tag);
	}
	virtual bool eval(const std::set<std::string>& tags) const
	{
		if (_tag[0] == '*')
		{
			string match = _tag.substr(1);
			for (std::set<string>::const_iterator i = tags.begin();
					i != tags.end(); i++)
				if (i->size() >= match.size() &&
						i->substr(i->size() - match.size()) == match)
					return true;
			return false;
		}
		else if (_tag[_tag.size() - 1] == '*')
		{
			string match = _tag.substr(0, _tag.size() - 1);
			for (std::set<string>::const_iterator i = tags.begin();
					i != tags.end(); i++)
				if (i->size() >= match.size() &&
						i->substr(0, match.size()) == match)
					return true;
			return false;
		}
		else
			return tags.find(_tag) != tags.end();
	}
};

class TagexprNot : public ExpressionImpl
{
protected:
	Expression _op;

public:
	TagexprNot(const Expression& operand) : _op(operand) {}

	virtual std::string format() const { return "!" + _op.format(); }

	virtual bool eval(const TagexprContext& context) const
	{
		return ! _op(context);
	}
	virtual bool eval(const std::set<std::string>& tags) const
	{
		return ! _op(tags);
	}
};

class TagexprAnd : public ExpressionImpl
{
protected:
	Expression _op1;
	Expression _op2;

public:
	TagexprAnd(const Expression& operand1, const Expression& operand2)
		: _op1(operand1), _op2(operand2) {}

	virtual std::string format() const
	{
		return "( " + _op1.format() + " && " + _op2.format() + " )";
	}

	virtual bool eval(const TagexprContext& context) const
	{
		return _op1(context) && _op2(context);
	}
	virtual bool eval(const std::set<std::string>& tags) const
	{
		return _op1(tags) && _op2(tags);
	}
};

class TagexprOr : public ExpressionImpl
{
protected:
	Expression _op1;
	Expression _op2;

public:
	TagexprOr(const Expression& operand1, const Expression& operand2)
		: _op1(operand1), _op2(operand2) {}
	
	virtual std::string format() const
	{
		return "( " + _op1.format() + " || " + _op2.format() + " )";
	}

	virtual bool eval(const TagexprContext& context) const
	{
		return _op1(context) || _op2(context);
	}
	virtual bool eval(const std::set<std::string>& tags) const
	{
		return _op1(tags) || _op2(tags);
	}
};


Expression::Expression(const std::string& expr) : m_impl(0)
{
	Expression e = TagexprParser::instance()->parse(expr);
	
	m_impl = e.impl();
	m_impl->ref();
}

Expression::Expression() : m_impl(new TagexprTrue) { m_impl->ref(); }

Expression Expression::matchTag(const std::string& pattern)
{
	return Expression(new TagexprTag(pattern));
}

Expression Expression::operator and (const Expression& e)
{
	return Expression(new TagexprAnd(*this, e));
}

Expression Expression::operator or (const Expression& e)
{
	return Expression(new TagexprOr(*this, e));
}

Expression Expression::operator not ()
{
	return Expression(new TagexprNot(*this));
}


bool TagexprContext::eval(const std::string& tag) const
{
	std::map<std::string, Expression>::const_iterator i = derivedTags.find(tag);
	if (i == derivedTags.end())
		return tags.find(tag) != tags.end();
	else if (seen.find(tag) == seen.end())
	{
		seen.insert(tag);
		bool res = i->second(*this);
		seen.erase(tag);
		return res;
	}
	else
		// If we got here, we have a circular dependency between derived tags
		// FIXME: find a better way to handle it: throw an exception, maybe?
		return false;
}

}

// vim:set ts=4 sw=4:
