/*
 * Implementation of some test utility functions
 *
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifndef TAGCOLL_TEST_UTILS_TCC
#define TAGCOLL_TEST_UTILS_TCC

#include <tests/test-utils.h>
#include <tagcoll/patch.h>
#include <tagcoll/coll/simple.h>
#include <wibble/operators.h>

namespace tagcoll {
namespace tests {

using namespace std;
using namespace tagcoll;
using namespace wibble::operators;
using namespace wibble::tests;
	
// Quick way of building a singleton string from a string literal
static inline wibble::Singleton<string> tss(const std::string& str)
{
	return wibble::singleton(str);
}

static inline wibble::Empty<string> empty()
{
	return wibble::Empty<string>();
}

template<typename COLL1, typename COLL2>
void __tc_ensure_coll_equals(const wibble::tests::Location& loc, const COLL1& c1, const COLL2& c2)
{
	PatchList<string, string> p;
	p.addPatch(c1, c2);
	if (!p.empty())
	{
		string f = "collections differ.  Patch:\n";
		for (PatchList<string, string>::const_iterator i = p.begin();
				i != p.end(); i++)
		{
			f += i->first;
			f += ": ";
			bool first = true;
			for (std::set<string>::const_iterator j = i->second.added.begin();
					j != i->second.added.end(); j++)
				if (first)
				{
					f += "+" + *j;
					first = false;
				} else {
					f += ", +" + *j;
				}
			for (std::set<string>::const_iterator j = i->second.removed.begin();
					j != i->second.removed.end(); j++)
				if (first)
				{
					f += "-" + *j;
					first = false;
				} else {
					f += ", -" + *j;
				}
			f += "\n";
		}
		
		throw tut::failure(loc.msg(f));
	}
}

template<typename OUT>
void __output_test_collection(const wibble::tests::Location& loc, OUT tc)
{
	std::set<string> tagset;

	tagset.clear();
	*tc = make_pair(tss("gnocco"), wibble::Empty<std::string>());
	++tc;

	tagset.clear();
	tagset.insert("tomato"); tagset.insert("mozzarella");
	*tc = make_pair(tss("margherita"), tagset);
	++tc;

	tagset.clear();
	tagset.insert("tomato"); tagset.insert("mozzarella"); tagset.insert("mushrooms");
	*tc = make_pair(tss("funghi"), tagset);
	++tc;

	tagset.clear();
	tagset.insert("garlic"); tagset.insert("rosemerry");
	*tc = make_pair(tss("rosmarino"), tagset);
	++tc;

	tagset.clear();
	tagset.insert("garlic"); tagset.insert("tomato");
	*tc = make_pair(tss("marinara"), tagset);
	++tc;
}

template<typename ROCOLL>
void __test_readonly_collection(const wibble::tests::Location& loc, const ROCOLL& tc)
{
	std::set<string> s, s1;

	// Try one iteration of the collection
	s.clear(); s1.clear();
	for (typename ROCOLL::const_iterator i = tc.begin();
			i != tc.end(); ++i)
	{
		s |= i->first;
		s1 |= i->second;
		/*
		cerr << i->first << ": ";
		for (typename ROCOLL::value_type::second_type::const_iterator j = i->second.begin(); j != i->second.end(); ++j)
			cerr << *j << ' ';
		cerr << endl;
		cerr << "s: ";
		for (std::set<string>::const_iterator j = s.begin(); j != s.end(); ++j)
			cerr << *j << ' ';
		cerr << endl;
		*/
	}
	inner_ensure_equals(s.size(), 4u);
	inner_ensure_equals(s1.size(), 5u);
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_contains(s, string("margherita"));
	inner_ensure_contains(s, string("funghi"));
	inner_ensure_contains(s, string("rosmarino"));
	inner_ensure_contains(s, string("marinara"));
	inner_ensure_contains(s1, string("tomato"));
	inner_ensure_contains(s1, string("mozzarella"));
	inner_ensure_contains(s1, string("mushrooms"));
	inner_ensure_contains(s1, string("garlic"));
	inner_ensure_contains(s1, string("rosemerry"));

	// hasTag
	inner_ensure(tc.hasTag("tomato"));
	inner_ensure(tc.hasTag("mozzarella"));
	inner_ensure(tc.hasTag("mushrooms"));
	inner_ensure(tc.hasTag("garlic"));
	inner_ensure(tc.hasTag("rosemerry"));
	inner_ensure(!tc.hasTag("ketchup"));

	// getTagsOfItem(item)
	s = tc.getTagsOfItem("funghi");
	inner_ensure_contains(s, string("tomato"));
	inner_ensure_contains(s, string("mozzarella"));
	inner_ensure_contains(s, string("mushrooms"));
	inner_ensure_not_contains(s, string("garlic"));
	inner_ensure_not_contains(s, string("rosemerry"));

	s = tc.getTagsOfItem("margherita");
	inner_ensure_contains(s, string("tomato"));
	inner_ensure_contains(s, string("mozzarella"));
	inner_ensure_not_contains(s, string("mushrooms"));
	inner_ensure_not_contains(s, string("garlic"));
	inner_ensure_not_contains(s, string("rosemerry"));

	s = tc.getTagsOfItem("rosmarino");
	inner_ensure_not_contains(s, string("tomato"));
	inner_ensure_not_contains(s, string("mozzarella"));
	inner_ensure_not_contains(s, string("mushrooms"));
	inner_ensure_contains(s, string("garlic"));
	inner_ensure_contains(s, string("rosemerry"));

	s = tc.getTagsOfItem("marinara");
	inner_ensure_contains(s, string("tomato"));
	inner_ensure_not_contains(s, string("mozzarella"));
	inner_ensure_not_contains(s, string("mushrooms"));
	inner_ensure_contains(s, string("garlic"));
	inner_ensure_not_contains(s, string("rosemerry"));

	s = tc.getTagsOfItem("gnocco");
	inner_ensure(s.empty());

	// getTagsOfitems(items)
	s1.clear();
	s1.insert("funghi"); s1.insert("margherita");
	s = tc.getTagsOfItems(s1);
	inner_ensure_contains(s, string("tomato"));
	inner_ensure_contains(s, string("mozzarella"));
	inner_ensure_contains(s, string("mushrooms"));
	inner_ensure_not_contains(s, string("garlic"));
	inner_ensure_not_contains(s, string("rosemerry"));

	s1.clear();
	s1.insert("rosmarino"); s1.insert("margherita");
	s = tc.getTagsOfItems(s1);
	inner_ensure_contains(s, string("tomato"));
	inner_ensure_contains(s, string("mozzarella"));
	inner_ensure_not_contains(s, string("mushrooms"));
	inner_ensure_contains(s, string("garlic"));
	inner_ensure_contains(s, string("rosemerry"));

	s1.clear();
	s1.insert("funghi"); s1.insert("margherita"); s1.insert("marinara");
	s = tc.getTagsOfItems(s1);
	inner_ensure_contains(s, string("tomato"));
	inner_ensure_contains(s, string("mozzarella"));
	inner_ensure_contains(s, string("mushrooms"));
	inner_ensure_contains(s, string("garlic"));
	inner_ensure_not_contains(s, string("rosemerry"));

	// getItemsHavingTag(tag)
	s = tc.getItemsHavingTag("tomato");
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_contains(s, string("margherita"));
	inner_ensure_contains(s, string("funghi"));
	inner_ensure_not_contains(s, string("rosmarino"));
	inner_ensure_contains(s, string("marinara"));

	s = tc.getItemsHavingTag("mozzarella");
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_contains(s, string("margherita"));
	inner_ensure_contains(s, string("funghi"));
	inner_ensure_not_contains(s, string("rosmarino"));
	inner_ensure_not_contains(s, string("marinara"));

	s = tc.getItemsHavingTag("mushrooms");
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_not_contains(s, string("margherita"));
	inner_ensure_contains(s, string("funghi"));
	inner_ensure_not_contains(s, string("rosmarino"));
	inner_ensure_not_contains(s, string("marinara"));

	s = tc.getItemsHavingTag("garlic");
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_not_contains(s, string("margherita"));
	inner_ensure_not_contains(s, string("funghi"));
	inner_ensure_contains(s, string("rosmarino"));
	inner_ensure_contains(s, string("marinara"));

	s = tc.getItemsHavingTag("rosemerry");
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_not_contains(s, string("margherita"));
	inner_ensure_not_contains(s, string("funghi"));
	inner_ensure_contains(s, string("rosmarino"));
	inner_ensure_not_contains(s, string("marinara"));

	// getItemsHavingTags(tags)
	s1.clear();
	s1.insert("tomato"); s1.insert("mozzarella");
	s = tc.getItemsHavingTags(s1);
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_contains(s, string("margherita"));
	inner_ensure_contains(s, string("funghi"));
	inner_ensure_not_contains(s, string("rosmarino"));
	inner_ensure_not_contains(s, string("marinara"));

	s1.clear();
	s1.insert("garlic"); s1.insert("rosemerry");
	s = tc.getItemsHavingTags(s1);
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_not_contains(s, string("margherita"));
	inner_ensure_not_contains(s, string("funghi"));
	inner_ensure_contains(s, string("rosmarino"));
	inner_ensure_not_contains(s, string("marinara"));

	s1.clear();
	s1.insert("tomato"); s1.insert("rosemerry");
	s = tc.getItemsHavingTags(s1);
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_not_contains(s, string("margherita"));
	inner_ensure_not_contains(s, string("funghi"));
	inner_ensure_not_contains(s, string("rosmarino"));
	inner_ensure_not_contains(s, string("marinara"));

	// TODO:    virtual std::set<ITEM> getTaggedItems() const = 0;
	// TODO:    virtual std::set<TAG> getAllTags() const = 0;
	// TODO:    virtual int getCardinality(const TAG& tag) const
	// TODO:    virtual std::set<TAG> getCompanionTags(const std::set<TAG>& tags) const

	// std::set<ITEM> getRelatedItems(const std::set<TAG>& tags, int maxdistance = 1) const
	s1.clear();
	s1.insert("tomato"); s1.insert("mozzarella");
	s = tc.getRelatedItems(s1, 1);
	inner_ensure_equals(s.size(), 2u);
	inner_ensure_not_contains(s, string("gnocco"));
	inner_ensure_contains(s, string("margherita"));
	inner_ensure_contains(s, string("funghi"));
	inner_ensure_not_contains(s, string("rosmarino"));
	inner_ensure_not_contains(s, string("marinara"));


	// void output(Consumer<ITEM, TAG>& consumer) const
	coll::Simple<string, string> coll1;
	tc.output(inserter(coll1));
	inner_ensure_coll_equals(tc, coll1);

	// TODO:    virtual void outputHavingTags(const std::set<TAG>& tags, Consumer<ITEM, TAG>& consumer) const

	std::vector<string> tags = tc.tagsInDiscriminanceOrder();
	inner_ensure_equals(tags.size(), 5u);
	inner_ensure(tags[4] != "rosemerry");
	inner_ensure(tags[4] != "mushrooms");
	inner_ensure(tags[0] != "tomato");
	inner_ensure(tags[0] != "mozzarella");
	inner_ensure(tags[0] != "garlic");

	coll1 = coll::Simple<string, string>();
	tc.output(inserter(coll1));

	s.clear();
	s.insert("rosemerry"); s.insert("olive oil");
	coll1.insert(tss("bianca"), s);

	s.clear();
	s.insert("garlic"); s.insert("mushrooms");
	coll1.insert(tss("trifolata"), s);

	tags = tc.tagsInRelevanceOrder(coll1);
	inner_ensure_equals(tags.size(), 5u);
	inner_ensure(tags[4] != "rosemerry");
	inner_ensure(tags[4] != "garlic");
	inner_ensure(tags[4] != "mushrooms");
	inner_ensure(tags[0] != "tomato");
	inner_ensure(tags[0] != "mozzarella");
}

template<typename COLL>
void __test_collection(const wibble::tests::Location& loc, COLL& tc)
{
	// Test handling of untagged items (they are not stored)
	tc.insert(tss("untagged"), wibble::Empty<string>());
	inner_ensure(tc.getTagsOfItem("untagged").empty());
	
	// Test handling of tagged items
	std::set<string> tagset;
	tagset.insert("tag1"); tagset.insert("tag2");
	tc.insert(tss("tagged"), tagset);
	inner_ensure_contains(tc.getTaggedItems(), string("tagged"));
	//inner_ensure(tc.hasTag("tag1"));
	//inner_ensure(tc.hasTag("tag2"));
	tagset = tc.getTagsOfItem("tagged");
	inner_ensure_contains(tagset, string("tag1"));
	inner_ensure_contains(tagset, string("tag2"));
	std::set<string> itemset = tc.getItemsHavingTag("tag1");
	inner_ensure_contains(itemset, string("tagged"));
	itemset = tc.getItemsHavingTag("tag2");
	inner_ensure_contains(itemset, string("tagged"));
	tagset = tc.getAllTags();
	inner_ensure_contains(tagset, string("tag1"));
	inner_ensure_contains(tagset, string("tag2"));
	tagset.clear();
	tagset.insert("tag1");
	tagset = tc.getCompanionTags(tagset);
	inner_ensure_not_contains(tagset, string("tag1"));
	inner_ensure_contains(tagset, string("tag2"));

	// Test handling of changes
	PatchList<string, string> change;
	Patch<string, string> p("tagged");
	p.remove("tag1");
	p.remove("tag2");
	change.addPatch(p);

	tc.applyChange(change);
	
	// "tagged" should now be untagged
	inner_ensure(tc.getTagsOfItem("tagged").empty());

	tc.applyChange(change.getReverse());

	// "tagged" should now be as before
	//inner_ensure(tc.hasTag("tag1"));
	//inner_ensure(tc.hasTag("tag2"));
	tagset = tc.getTagsOfItem("tagged");
	inner_ensure_contains(tagset, string("tag1"));
	inner_ensure_contains(tagset, string("tag2"));
	itemset = tc.getItemsHavingTag("tag1");
	inner_ensure_contains(itemset, string("tagged"));
	itemset = tc.getItemsHavingTag("tag2");
	inner_ensure_contains(itemset, string("tagged"));
	tagset = tc.getAllTags();
	inner_ensure_contains(tagset, string("tag1"));
	inner_ensure_contains(tagset, string("tag2"));
	tagset.clear();
	tagset.insert("tag1");
	tagset = tc.getCompanionTags(tagset);
	inner_ensure_not_contains(tagset, string("tag1"));
	inner_ensure_contains(tagset, string("tag2"));

	// Try a patch that adds a tag
	change = PatchList<string, string>();
	p = Patch<string, string>("tagged");
	p.add("tag3");
	change.addPatch(p);
	tc.applyChange(change);

	//inner_ensure(tc.hasTag("tag1"));
	//inner_ensure(tc.hasTag("tag2"));
	//inner_ensure(tc.hasTag("tag3"));
	tagset = tc.getTagsOfItem("tagged");
	inner_ensure_contains(tagset, string("tag1"));
	inner_ensure_contains(tagset, string("tag2"));
	inner_ensure_contains(tagset, string("tag3"));
	itemset = tc.getItemsHavingTag("tag1");
	inner_ensure_contains(itemset, string("tagged"));
	itemset = tc.getItemsHavingTag("tag2");
	inner_ensure_contains(itemset, string("tagged"));
	itemset = tc.getItemsHavingTag("tag3");
	inner_ensure_contains(itemset, string("tagged"));
	tagset = tc.getAllTags();
	inner_ensure_contains(tagset, string("tag1"));
	inner_ensure_contains(tagset, string("tag2"));
	inner_ensure_contains(tagset, string("tag3"));
	tagset.clear();
	tagset.insert("tag1");
	tagset = tc.getCompanionTags(tagset);
	inner_ensure_not_contains(tagset, string("tag1"));
	inner_ensure_contains(tagset, string("tag2"));
	inner_ensure_contains(tagset, string("tag3"));

	// Try a patch that adds some items
	change = PatchList<string, string>();
	p = Patch<string, string>("tagged1");
	p.add("tag1");
	p.add("tag2");
	p.add("tag4");
	change.addPatch(p);
	tc.applyChange(change);

	tagset = tc.getTagsOfItem("tagged1");
	inner_ensure_contains(tagset, string("tag1"));
	inner_ensure_contains(tagset, string("tag2"));
	inner_ensure_not_contains(tagset, string("tag3"));
	inner_ensure_contains(tagset, string("tag4"));
	itemset = tc.getItemsHavingTag("tag1");
	inner_ensure_contains(itemset, string("tagged1"));
	itemset = tc.getItemsHavingTag("tag2");
	inner_ensure_contains(itemset, string("tagged1"));
	itemset = tc.getItemsHavingTag("tag3");
	inner_ensure_not_contains(itemset, string("tagged1"));
	itemset = tc.getItemsHavingTag("tag4");
	inner_ensure_not_contains(itemset, string("tagged"));
	inner_ensure_contains(itemset, string("tagged1"));
	tagset = tc.getAllTags();
	inner_ensure_contains(tagset, string("tag1"));
	inner_ensure_contains(tagset, string("tag2"));
	inner_ensure_contains(tagset, string("tag3"));
	inner_ensure_contains(tagset, string("tag4"));
	tagset.clear();
	tagset.insert("tag1");
	tagset = tc.getCompanionTags(tagset);
	inner_ensure_not_contains(tagset, string("tag1"));
	inner_ensure_contains(tagset, string("tag2"));
	inner_ensure_contains(tagset, string("tag3"));
	inner_ensure_contains(tagset, string("tag4"));

	// And reverse it
	tc.applyChange(change.getReverse());

	itemset = tc.getItemsHavingTag("tag1");
	inner_ensure_not_contains(itemset, string("tagged1"));
	itemset = tc.getItemsHavingTag("tag2");
	inner_ensure_not_contains(itemset, string("tagged1"));
	itemset = tc.getItemsHavingTag("tag3");
	inner_ensure_not_contains(itemset, string("tagged1"));
	inner_ensure(tc.getItemsHavingTag("tag4") == std::set<string>());
	tagset = tc.getAllTags();
	inner_ensure_contains(tagset, string("tag1"));
	inner_ensure_contains(tagset, string("tag2"));
	inner_ensure_contains(tagset, string("tag3"));
	inner_ensure_not_contains(tagset, string("tag4"));
	tagset.clear();
	tagset.insert("tag1");
	tagset = tc.getCompanionTags(tagset);
	inner_ensure_not_contains(tagset, string("tag1"));
	inner_ensure_contains(tagset, string("tag2"));
	inner_ensure_contains(tagset, string("tag3"));
	inner_ensure_not_contains(tagset, string("tag4"));

	// Ensure that clear() is implemented
	tc.clear();
	itemset = tc.getTaggedItems();
	inner_ensure(itemset.empty());
	tagset = tc.getAllTags();
	inner_ensure(tagset.empty());
}

}
}

#include <tagcoll/patch.tcc>
#include <tagcoll/coll/base.tcc>

#endif

// vim:set ts=4 sw=4:
