/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2024 Univ. Grenoble Alpes, CNRS, Grenoble INP - UGA, TIMC, 38000 Grenoble, France
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "VRMLComponentExtension.h"

#include "VRMLComponent.h"

//-- CamiTK stuff
#include <Application.h>
#include <InteractiveGeometryViewer.h>
#include <Log.h>
using namespace camitk;

//-- vtk stuff
// disable warning generated by clang about the surrounded headers
#include <CamiTKDisableWarnings>
#include <vtkVRMLExporter.h>
#include <CamiTKReEnableWarnings>

// --------------- getName -------------------
QString VRMLComponentExtension::getName() const {
    return "VRML 2 Component";
}

// --------------- getDescription -------------------
QString VRMLComponentExtension::getDescription() const {
    return tr("Basic support for VRML 2 <em>.wrl</em> and <em>.vrml</em> documents");
}

// --------------- getFileExtensions -------------------
QStringList VRMLComponentExtension::getFileExtensions() const {
    QStringList ext;
    ext << "vrml" << "wrl";
    return ext;
}

// --------------- open -------------------
Component* VRMLComponentExtension::open(const QString& fileName) {
    return new VRMLComponent(fileName);
}

// --------------- save -------------------
bool VRMLComponentExtension::save(Component* component) const {
    // NOTE: in Vtk an exporter can only export all the scene, while a Vtk writer is able to save a single mesh
    // As there are no vtkVRMLWriter, the whole scene is exported (so close everything you don't
    // want to have in the resulting vrml file!)
    MeshComponent* mesh = dynamic_cast<MeshComponent*>(component);

    if (mesh == nullptr) {
        // should never occurs
        return false;
    }
    else {
        if (mesh->getPointSet()) {
            InteractiveGeometryViewer* default3DViewer = dynamic_cast<InteractiveGeometryViewer*>(Application::getViewer("3D Viewer"));

            if (default3DViewer != nullptr) {
                vtkSmartPointer<vtkVRMLExporter> exporter = vtkSmartPointer<vtkVRMLExporter>::New();
#if VTK_MAJOR_VERSION < 9
                exporter->SetInput(default3DViewer->getRendererWidget()->GetRenderWindow());
#else
                exporter->SetInput(default3DViewer->getRendererWidget()->renderWindow());
#endif
                exporter->SetFileName(mesh->getFileName().toStdString().c_str());
                exporter->Write();
                return true;
            }
            else {
                CAMITK_ERROR(tr("Cannot find \"3D Viewer\". This viewer is mandatory for exporting to VRML."))
                return false;
            }
        }
        else {
            return false;
        }
    }
}

